/** @file   GraphicsItem.cpp
 * @brief   Implementation of GraphicsItem class.
 * @version $Revision: 1.1 $
 * @date    $Date: 2006/02/21 20:48:36 $
 * @author  Tomi Lamminsaari
 */

#include "GraphicsContainer.h"
#include <allegro.h>

namespace eng2d {

///
/// Constants, datatypes and static methods
/// ============================================================================




///
/// Constructors, destructor and operators
/// ============================================================================

/** Default constructor.
 */
GraphicsContainer::GraphicsContainer( GfxContentType aContentType ) :
  iContentType( aContentType )
{
}



/** Destructor.
 */
GraphicsContainer::~GraphicsContainer()
{
  this->ResetAndDestroy();
}



///
/// Methods inhertited from the base class(es)
/// ============================================================================

/** Empties this item.
 */
void GraphicsContainer::ResetAndDestroy()
{
  for ( int i=0; i < iGraphics.size(); i++ ) {
    void* gfx = iGraphics.at(i);
    switch ( iContentType ) {
      case ( ESprite ):
      case ( EBitmap ): {
        destroy_bitmap( reinterpret_cast<BITMAP*>(gfx) );
        break;
      }
      case ( ERleSprite ): {
        destroy_rle_sprite( reinterpret_cast<RLE_SPRITE*>(gfx) );
        break;
      }
      case ( ECompiledSprite ): {
        destroy_compiled_sprite( reinterpret_cast<COMPILED_SPRITE*>(gfx) );
        break;
      }
    }
    iGraphics.at(i) = 0;
  }
  iGraphics.clear();
  iContentType = EUnspecified;
}


/** Appends data to this item. 
 */
void GraphicsContainer::Push( void* aObject )
{
  iGraphics.push_back( aObject );
}


/** Returns the index'th item.
 */
void* GraphicsContainer::At( int aUid ) const
{
  if ( aUid < 0 || aUid >= iGraphics.size() ) {
    return 0;
  }
  return iGraphics.at( aUid );
}




///
/// New public methods
/// ============================================================================




///
/// Getter methods
/// ============================================================================

/** Returns the index'th bitmap.
 */
BITMAP* GraphicsContainer::GetBitmap( int aIndex ) const throw ( xOutOfBounds,
                                                                 xLogicError )
{
  if ( iContentType != EBitmap ) {
    throw xLogicError( "GraphicsItem",
                       "GetBitmap(int aIndex)",
                       "Invalid content type" );
  }
  if ( aIndex < 0 || aIndex >= iGraphics.size() ) {
    throw xOutOfBounds( "GraphicsItem",
                        "GetBitmap(int aIndex)" );
  }
  return reinterpret_cast<BITMAP*>( iGraphics.at(aIndex) );
}



/** Returns the index'th rle sprite.
 */
RLE_SPRITE* GraphicsContainer::GetRleSprite( int aIndex ) const throw( xOutOfBounds,
                                                                       xLogicError )
{
  if ( iContentType != ERleSprite ) {
    throw xLogicError( "GraphicsItem",
                       "GetBitmap(int aIndex)",
                       "Invalid content type" );
  }
  if ( aIndex < 0 || aIndex >= iGraphics.size() ) {
    throw xOutOfBounds( "GraphicsItem",
                        "GetBitmap(int aIndex)" );
  }
  return reinterpret_cast<RLE_SPRITE*>( iGraphics.at(aIndex) );
}


/** Returns the content type information.
 */
GfxContentType GraphicsContainer::ContentType() const
{
  return iContentType;
}



/** Returns the number of graphic objects this item contains.
 */
int GraphicsContainer::GraphicCount() const
{
  return iGraphics.size();
}

///
/// Protected and private methods
/// ============================================================================


};  // end of namespace
